import { useState, useRef, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Scissors, Ruler, Palette, Sparkles, Calendar, CheckCircle, Zap, Target, Award, Star, Users, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedLevel, setSelectedLevel] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const levels = [
    { 
      id: 'basic', 
      name: 'Базовый', 
      icon: Scissors,
      color: '#FF6B6B',
      description: 'Основы шитья для начинающих. Научитесь работать с тканью и швейной машинкой.',
      duration: '6 недель',
      students: '800+',
      rating: '4.9'
    },
    { 
      id: 'cutting', 
      name: 'Кройка', 
      icon: Ruler,
      color: '#4ECDC4',
      description: 'Изучите техники раскроя ткани и создание выкроек для различных моделей.',
      duration: '8 недель',
      students: '650+',
      rating: '4.8'
    },
    { 
      id: 'modeling', 
      name: 'Моделирование', 
      icon: Palette,
      color: '#9B59B6',
      description: 'Создание собственных моделей одежды и работа с формой и силуэтом.',
      duration: '10 недель',
      students: '500+',
      rating: '4.9'
    },
    { 
      id: 'author', 
      name: 'Авторская одежда', 
      icon: Sparkles,
      color: '#FFA726',
      description: 'Разработка уникальных коллекций и создание авторских моделей одежды.',
      duration: '12 недель',
      students: '350+',
      rating: '5.0'
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedLevel || !startDate) {
      alert('Пожалуйста, выберите уровень и дату начала')
      return
    }

    const saved = localStorage.getItem('sewingCourses')
    const courses = saved ? JSON.parse(saved) : []
    
    const levelData = levels.find(l => l.id === selectedLevel)
    
    // Находим максимальный ID и добавляем 1
    const maxId = courses.length > 0 ? Math.max(...courses.map(c => c.id)) : 0
    const newId = maxId + 1
    
    const newCourse = {
      id: newId,
      project: `Проект ${levelData.name}`,
      teacher: 'Преподаватель будет назначен',
      completionDate: startDate,
      status: 'active',
      resultPhoto: null
    }
    
    courses.push(newCourse)
    localStorage.setItem('sewingCourses', JSON.stringify(courses))
    
    setIsSubmitted(true)
    
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  return (
    <div className="enroll-page-modern">
      <div className="container">
        <section className="enroll-header-modern">
          <div className="header-badge-modern">
            <Scissors size={18} />
            <span>ЗАПИСЬ НА КУРС</span>
          </div>
          <h1 className="enroll-title-modern">
            Записаться на <span className="title-accent-modern">курс</span>
          </h1>
          <p className="enroll-description-modern">
            Выберите уровень обучения и дату начала. Все курсы включают практические занятия и обратную связь от мастеров.
          </p>
        </section>

        <section className="enroll-form-section-modern">
          <form className="form-modern" onSubmit={handleSubmit}>
            <div className="levels-header-modern">
              <h2 className="levels-title-modern">Выберите уровень обучения</h2>
              <p className="levels-description-modern">
                Каждый уровень рассчитан на определенный опыт и навыки
              </p>
            </div>
            <div className="levels-selection-modern">
              {levels.map((level) => {
                const Icon = level.icon
                return (
                  <button
                    key={level.id}
                    type="button"
                    className={`level-card-modern ${selectedLevel === level.id ? 'selected' : ''}`}
                    onClick={() => setSelectedLevel(level.id)}
                    style={{ '--level-color': level.color }}
                  >
                    <div className="level-icon-wrapper-modern">
                      <Icon size={36} />
                    </div>
                    <h3 className="level-name-modern">{level.name}</h3>
                    <p className="level-desc-modern">{level.description}</p>
                    <div className="level-stats-modern">
                      <div className="level-stat-modern">
                        <span className="stat-label-modern">Длительность:</span>
                        <span className="stat-value-modern">{level.duration}</span>
                      </div>
                      <div className="level-stat-modern">
                        <span className="stat-label-modern">Студентов:</span>
                        <span className="stat-value-modern">{level.students}</span>
                      </div>
                      <div className="level-stat-modern">
                        <span className="stat-label-modern">Рейтинг:</span>
                        <span className="stat-value-modern">{level.rating}</span>
                      </div>
                    </div>
                    {selectedLevel === level.id && (
                      <div className="check-badge-modern">
                        <CheckCircle size={20} />
                      </div>
                    )}
                  </button>
                )
              })}
            </div>

            {selectedLevel && (
              <div className="date-section-modern">
                <div className="date-label-modern">
                  <Calendar size={18} />
                  <span>Дата начала курса</span>
                </div>
                <input
                  type="date"
                  className="date-input-modern"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={minDateStr}
                  max={maxDateStr}
                  required
                />
              </div>
            )}

            <button 
              type="submit" 
              className="submit-btn-modern" 
              disabled={!selectedLevel || !startDate || isSubmitted}
            >
              <span>Записаться на курс</span>
            </button>

            {isSubmitted && (
              <div className="success-message-modern">
                <CheckCircle size={28} />
                <span>Запись успешно оформлена!</span>
              </div>
            )}
          </form>
        </section>

        <ImageSlider />

        <InstructorsSection />

        <FeaturesSection />
      </div>
    </div>
  )
}

const InstructorsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const instructors = [
    {
      name: 'Елена Иванова',
      specialization: 'Классическое шитье',
      experience: '15 лет опыта',
      students: '500+',
      image: 'linear-gradient(135deg, #FF6B6B 0%, #4ECDC4 100%)',
      rating: 4.9
    },
    {
      name: 'Мария Петрова',
      specialization: 'Дизайн одежды',
      experience: '12 лет опыта',
      students: '380+',
      image: 'linear-gradient(135deg, #9B59B6 0%, #FF6B6B 100%)',
      rating: 5.0
    },
    {
      name: 'Анна Соколова',
      specialization: 'Авторская одежда',
      experience: '18 лет опыта',
      students: '620+',
      image: 'linear-gradient(135deg, #4ECDC4 0%, #FFA726 100%)',
      rating: 4.8
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`instructors-section-modern ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="instructors-header-modern">
          <h2 className="instructors-title-modern">Наши преподаватели</h2>
          <p className="instructors-description-modern">
            Опытные мастера с многолетним стажем
          </p>
        </div>
        <div className="instructors-grid-modern">
          {instructors.map((instructor, index) => (
            <div 
              key={index} 
              className="instructor-card-modern"
              style={{ 
                animationDelay: `${index * 0.15}s`,
                background: instructor.image
              }}
            >
              <div className="instructor-image-modern"></div>
              <div className="instructor-content-modern">
                <div className="instructor-rating-modern">
                  <Star size={18} fill="#FFBE0B" color="#FFBE0B" />
                  <span>{instructor.rating}</span>
                </div>
                <h3 className="instructor-name-modern">{instructor.name}</h3>
                <div className="instructor-specialization-modern">{instructor.specialization}</div>
                <div className="instructor-stats-modern">
                  <div className="instructor-stat-modern">
                    <Clock size={14} />
                    <span>{instructor.experience}</span>
                  </div>
                  <div className="instructor-stat-modern">
                    <Users size={14} />
                    <span>{instructor.students} студентов</span>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

const FeaturesSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const features = [
    {
      icon: Zap,
      title: 'Быстрый старт',
      description: 'Начните создавать уже с первого занятия',
      color: 'var(--modern-accent)'
    },
    {
      icon: Target,
      title: 'Четкие цели',
      description: 'Понятная программа и структурированное обучение',
      color: 'var(--modern-teal)'
    },
    {
      icon: Award,
      title: 'Сертификат',
      description: 'Получите подтверждение ваших навыков',
      color: 'var(--modern-purple)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`features-section-modern ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="features-header-modern">
          <h2 className="features-title-modern">Особенности</h2>
          <p className="features-description-modern">
            Что делает наши курсы уникальными
          </p>
        </div>
        <div className="features-grid-modern">
          {features.map((feature, index) => {
            const Icon = feature.icon
            return (
              <div 
                key={index} 
                className="feature-card-modern"
                style={{ 
                  animationDelay: `${index * 0.1}s`,
                  '--feature-color': feature.color
                }}
              >
                <div className="feature-icon-wrapper-modern">
                  <Icon size={28} />
                </div>
                <h3 className="feature-title-modern">{feature.title}</h3>
                <p className="feature-text-modern">{feature.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

export default Enroll

